<?php
require_once __DIR__ . '/../includes/_init.php';
require_auth();
post_only();
csrf_verify();

require_once __DIR__ . '/../../app/services/GeoService.php';
require_once __DIR__ . '/../../app/services/PlaceService.php';
require_once __DIR__ . '/../../app/services/RouteService.php';
require_once __DIR__ . '/../../app/services/TripEngine.php';

$pdo = db();

try {
    $tripName = trim($_POST['trip_name'] ?? 'My Trip');
    $city = trim($_POST['city'] ?? '');
    $startTime = $_POST['start_time'] ?? '10:00';
    $endTime = $_POST['end_time'] ?? '18:00';
    $pace = $_POST['pace'] ?? 'normal';
    $budget = $_POST['budget'] ?? null;
    $visitMinutes = (int)($_POST['visit_minutes'] ?? 60);
    $maxPlaces = (int)($_POST['max_places'] ?? 8);
    $avoidLong = !empty($_POST['avoid_long_travel']) ? 1 : 0;
    $startLocation = trim($_POST['start_location'] ?? '');
    $mood = trim($_POST['mood'] ?? '');

    if ($city === '') throw new Exception('City is required.');
    if ($maxPlaces < 5) $maxPlaces = 5;
    if ($maxPlaces > MAX_PLACES_CAP) $maxPlaces = MAX_PLACES_CAP;

    // 1) City validation
    $cityGeo = GeoService::geocodeCity($city);

    // 2) Start point (optional)
    if ($startLocation !== '') {
        $startGeo = GeoService::geocodeLocation($startLocation);
    } else {
        $startGeo = [
            'display_name' => $cityGeo['display_name'],
            'lat' => $cityGeo['lat'],
            'lng' => $cityGeo['lng'],
        ];
    }

    // 3) Categories: explicit + mood mapping
    $selectedCategoryIds = $_POST['categories'] ?? [];
    $selectedCategoryIds = array_map('intval', $selectedCategoryIds);

    if ($mood !== '') {
        $moodSlugs = TripEngine::parseMoodToInterests($mood);
        if (!empty($moodSlugs)) {
            $in = str_repeat('?,', count($moodSlugs)-1) . '?';
            $st = $pdo->prepare("SELECT id FROM categories WHERE slug IN ($in) AND is_active=1");
            $st->execute($moodSlugs);
            foreach ($st->fetchAll() as $r) $selectedCategoryIds[] = (int)$r['id'];
        }
    }
    $selectedCategoryIds = array_values(array_unique($selectedCategoryIds));

    if (empty($selectedCategoryIds)) {
        throw new Exception('Please select at least one interest category OR write a mood description.');
    }

    $in = str_repeat('?,', count($selectedCategoryIds)-1) . '?';
    $stCats = $pdo->prepare("SELECT * FROM categories WHERE id IN ($in) AND is_active=1");
    $stCats->execute($selectedCategoryIds);
    $cats = $stCats->fetchAll();
    if (empty($cats)) throw new Exception('Selected categories are not available.');

    // 4) Fetch places by category
    $allPlaces = [];
    $radius = DEFAULT_RADIUS_METERS;
    foreach ($cats as $c) {
        $otmKind = $c['otm_kind'];
        $found = PlaceService::fetchByCategory($cityGeo['lat'], $cityGeo['lng'], $otmKind, $radius, PLACES_PER_CATEGORY);
        foreach ($found as $p) {
            $p['category_id'] = (int)$c['id'];
            $p['category_name'] = $c['name'];
            $allPlaces[] = $p;
        }
    }

    $allPlaces = TripEngine::dedupeByExternalId($allPlaces);

    // 5) Filters + scoring
    $scored = [];
    foreach ($allPlaces as $p) {
        if (empty($p['lat']) || empty($p['lng'])) continue;

        $distKm = TripEngine::haversineKm($startGeo['lat'], $startGeo['lng'], $p['lat'], $p['lng']);
        if ($avoidLong && $distKm > 10) continue;

        // rating filter if present
        if (!empty($p['rating']) && (float)$p['rating'] < 3.5) continue;

        $score = TripEngine::scorePlace($p, [
            'interest_match' => true,
            'dist_km' => $distKm,
        ]);
        $p['score'] = $score;
        $p['dist_km'] = $distKm;
        $scored[] = $p;
    }

    if (empty($scored)) throw new Exception('No good places found. Try changing categories or constraints.');

    usort($scored, fn($a,$b) => ($b['score'] <=> $a['score']));
    $selected = array_slice($scored, 0, $maxPlaces);

    // 6) Persist trip
    $pdo->beginTransaction();

    $insTrip = $pdo->prepare("INSERT INTO trips (user_id,trip_name,city,city_display,city_lat,city_lng,start_label,start_lat,start_lng,start_time,end_time,pace,budget,max_places,avoid_long_travel,visit_minutes,created_at)
                              VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
    $insTrip->execute([
        (int)$_SESSION['user_id'],
        $tripName,
        $city,
        $cityGeo['display_name'],
        $cityGeo['lat'], $cityGeo['lng'],
        $startGeo['display_name'],
        $startGeo['lat'], $startGeo['lng'],
        $startTime . ":00",
        $endTime . ":00",
        $pace,
        $budget ?: null,
        $maxPlaces,
        $avoidLong,
        $visitMinutes,
        now(),
    ]);
    $tripId = (int)$pdo->lastInsertId();

    $insPref = $pdo->prepare("INSERT INTO trip_preferences (trip_id,category_id) VALUES (?,?)");
    foreach ($cats as $c) $insPref->execute([$tripId, (int)$c['id']]);

    // Upsert places_master + insert trip_places
    $selPlace = $pdo->prepare("SELECT id FROM places_master WHERE external_place_id=? LIMIT 1");
    $insPlace = $pdo->prepare("INSERT INTO places_master (external_place_id,name,address,lat,lng,rating,photo_url,created_at) VALUES (?,?,?,?,?,?,?,?)");
    $updPlace = $pdo->prepare("UPDATE places_master SET name=?, lat=?, lng=?, rating=? WHERE id=?");

    $insTripPlace = $pdo->prepare("INSERT INTO trip_places (trip_id,place_id,category_id,score,selected_flag) VALUES (?,?,?,?,1)");

    $placeIdByExternal = [];
    foreach ($selected as $p) {
        $xid = $p['external_place_id'];
        $selPlace->execute([$xid]);
        $row = $selPlace->fetch();

        if ($row) {
            $pid = (int)$row['id'];
            $updPlace->execute([$p['name'], $p['lat'], $p['lng'], $p['rating'], $pid]);
        } else {
            $insPlace->execute([$xid, $p['name'], null, $p['lat'], $p['lng'], $p['rating'], null, now()]);
            $pid = (int)$pdo->lastInsertId();
        }
        $placeIdByExternal[$xid] = $pid;

        $insTripPlace->execute([$tripId, $pid, (int)$p['category_id'], (float)$p['score']);
    }

    $pdo->commit();

    // 7) Build route + schedule (feasibility loop)
    $availSeconds = strtotime($endTime) - strtotime($startTime);
    if ($availSeconds <= 0) throw new Exception('End time must be later than start time.');

    // Load selected places with ids
    $stPl = $pdo->prepare("SELECT tp.place_id, pm.external_place_id, pm.name, pm.lat, pm.lng, tp.score
                           FROM trip_places tp
                           JOIN places_master pm ON pm.id=tp.place_id
                           WHERE tp.trip_id=? AND tp.selected_flag=1
                           ORDER BY tp.score DESC");
    $stPl->execute([$tripId]);
    $places = $stPl->fetchAll();

    $dropped = 0;
    while (count($places) > 1) {
        // coords: start + places (in score order initially)
        $idxToPlaceId = [0 => 'START'];
        $coords = [[(float)$startGeo['lng'], (float)$startGeo['lat']]];
        $i = 1;
        foreach ($places as $pl) {
            $idxToPlaceId[$i] = (int)$pl['place_id'];
            $coords[] = [(float)$pl['lng'], (float)$pl['lat']];
            $i++;
        }

        $mx = RouteService::matrix($coords);
        $dur = $mx['durations'] ?? null;
        $dist = $mx['distances'] ?? null;
        if (!$dur || !$dist) throw new Exception('Matrix API returned incomplete data.');

        $orderIdx = TripEngine::nearestNeighborOrder($dur); // includes 0 first
        // convert to place_id order (exclude 0)
        $routePlaceIds = [];
        for ($k=1; $k<count($orderIdx); $k++) {
            $routePlaceIds[] = $idxToPlaceId[$orderIdx[$k]];
        }

        // compute total travel (order path)
        $totalTravel = 0;
        $totalDistance = 0;
        for ($k=0; $k<count($orderIdx)-1; $k++) {
            $a = $orderIdx[$k];
            $b = $orderIdx[$k+1];
            $totalTravel += (float)($dur[$a][$b] ?? 0);
            $totalDistance += (float)($dist[$a][$b] ?? 0);
        }
        $totalVisit = count($routePlaceIds) * ($visitMinutes * 60);
        $total = $totalTravel + $totalVisit;

        if ($total <= $availSeconds) {
            // Directions geojson for map
            // Build ordered coords for directions: start + in route order
            $orderedCoords = [[(float)$startGeo['lng'], (float)$startGeo['lat']]];
            foreach ($orderIdx as $idx) {
                if ($idx == 0) continue;
                $orderedCoords[] = $coords[$idx];
            }
            $geojson = RouteService::directions($orderedCoords);

            // Persist route + schedule
            $pdo->beginTransaction();
            $pdo->prepare("DELETE FROM trip_route WHERE trip_id=?")->execute([$tripId]);
            $pdo->prepare("DELETE FROM trip_schedule WHERE trip_id=?")->execute([$tripId]);

            $insR = $pdo->prepare("INSERT INTO trip_route (trip_id, route_order_json, total_distance_m, total_time_s, geojson, created_at)
                                   VALUES (?,?,?,?,?,?)");
            $insR->execute([$tripId, json_encode($routePlaceIds), $totalDistance, $totalTravel, json_encode($geojson), now()]);

            $schedule = TripEngine::buildSchedule($startTime, $endTime, $routePlaceIds, $visitMinutes, $dur, $idxToPlaceId);
            $insS = $pdo->prepare("INSERT INTO trip_schedule (trip_id, schedule_json, created_at) VALUES (?,?,?)");
            $insS->execute([$tripId, json_encode($schedule), now()]);

            $pdo->commit();

            set_flash('success', 'Trip generated successfully' . ($dropped ? " (reduced places to fit time)." : "."));
            redirect('/user/trip_view.php?id=' . $tripId);
        }

        // Drop the lowest scored place and retry
        array_pop($places);
        $dropped++;
    }

    throw new Exception('Time window is too small to fit even 1 place. Increase end time or reduce visit duration.');

} catch (Exception $ex) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    set_flash('danger', $ex->getMessage());
    redirect('/user/create_trip.php');
}
