<?php
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../config.php';

class PlaceService {
    // OpenTripMap documentation: https://opentripmap.io/product
    public static function fetchByCategory(float $lat, float $lng, string $otmKind, int $radiusMeters, int $limit): array {
        if (!OTM_API_KEY || OTM_API_KEY === 'YOUR_OPENTRIPMAP_API_KEY') {
            throw new Exception('OpenTripMap API key missing. Please set OTM_API_KEY in app/config.php');
        }

        $kinds = urlencode($otmKind);
        $url = "https://api.opentripmap.com/0.1/en/places/radius?radius={$radiusMeters}&lon={$lng}&lat={$lat}&kinds={$kinds}&format=json&limit={$limit}&apikey=" . urlencode(OTM_API_KEY);
        $rows = http_json($url);

        // Each row has xid, name, dist, rate, kinds, osm, wikidata, point{lat,lon}
        $places = [];
        foreach ($rows as $r) {
            $p = $r['point'] ?? null;
            if (!$p || !isset($p['lat']) || !isset($p['lon'])) continue;
            $name = trim((string)($r['name'] ?? ''));
            if ($name === '') continue;

            $places[] = [
                'external_place_id' => $r['xid'] ?? null,
                'name' => $name,
                'lat' => (float)$p['lat'],
                'lng' => (float)$p['lon'],
                'address' => '', // will be filled from details call if needed
                'rating' => isset($r['rate']) ? (float)$r['rate'] : null,
                'photo_url' => null,
                'raw_kinds' => $r['kinds'] ?? '',
                'dist_m' => isset($r['dist']) ? (float)$r['dist'] : null,
            ];
        }
        return $places;
    }

    public static function details(string $xid): array {
        if (!OTM_API_KEY || OTM_API_KEY === 'YOUR_OPENTRIPMAP_API_KEY') {
            throw new Exception('OpenTripMap API key missing. Please set OTM_API_KEY in app/config.php');
        }
        $url = "https://api.opentripmap.com/0.1/en/places/xid/" . urlencode($xid) . "?apikey=" . urlencode(OTM_API_KEY);
        $d = http_json($url);

        $addr = '';
        if (isset($d['address']) && is_array($d['address'])) {
            $a = $d['address'];
            $parts = [];
            foreach (['road','suburb','city','state','postcode','country'] as $k) {
                if (!empty($a[$k])) $parts[] = $a[$k];
            }
            $addr = implode(', ', $parts);
        }

        return [
            'address' => $addr,
            'photo_url' => $d['preview']['source'] ?? null,
            'wikipedia' => $d['wikipedia'] ?? null,
            'website' => $d['url'] ?? null,
        ];
    }
}
